
/** 
  * Module for monitoring Outgoing Traffic, Mobility among the nodes and Channel assesments
  *
  * @author Wasif Masood
  * @date   July 11th, 2010
  */ 


#include "printf.h"

generic module AnalyzerP(layer_id id){

	provides interface PacketLoad;
	provides interface RadioAnalyzer;

  uses interface Timer<TMilli> as OnTimer;			
	uses interface LocalTime<TMilli>;
	uses interface CC2420PacketBody;
	uses interface Packet;

} 
 
implementation{
	
	uint16_t m_rate = 1000;				/*monitoring rate in milli sec for outbound traffics*/
	uint32_t total_bytes=0;

	float alpha = 0.6;
/*	float actual=0;
	float forecast=0;*/

	float sentRate = 0;
	float receiveRate = 0;
	float dropRate = 0;
	float throughputRate=0;

	uint32_t packetsSend = 0;
	uint32_t packetsLost = 0;	


	uint32_t send_bytes = 0;				// for Tx loging
	uint32_t receive_bytes = 0;			// for Rx logging

	uint32_t sent_bytes = 0;				// for actual throughput logging


/** Control Variables for Packet Delay**/

	struct PacketDelayInfo {
		uint16_t dsn;	
		uint8_t frameType;
		uint16_t pSendTime;
		uint16_t pSendDoneTime;
		error_t error;
	};

	struct 

	struct PacketDelayInfo PacketDelay;

/* Control variables for RadioAnalyzer*/
	uint32_t rStartTime = 0;
	uint32_t rStopTime = 0;
	
	uint32_t rTicOff = 0;
	uint32_t rTocOff = 0;

	uint64_t rOnTime = 0;
	uint64_t rOffTime = 0;


/* Local Functions */
	void refreshTxDataRate();
	void refreshRxDataRate();
	void refreshDropRate();
	void refreshThroughput();

	/************************************************************************/
	/***************************** RadioAnalyer *****************************/
	/************************************************************************/
	
	command void RadioAnalyzer.logStart() 
	{

		uint32_t diff = 0;
		
		rStartTime = call LocalTime.get();

		//printf("\n %ld", rStartTime);printfflush();

		rTocOff = rStartTime;
		
		if(rTicOff!=0)
		{
			diff = rTocOff - rTicOff;
			rOffTime = rOffTime + diff;
		}	

	}

	command void RadioAnalyzer.logStop() {

		uint32_t diff = 0;

		rStopTime = call LocalTime.get();
		rTicOff= rStopTime;

		/** Condition Check for Timer overflow **/
		/*
		if(rStartTime > rStopTime){
			diff = 65534L - rStartTime;
			diff+= rStopTime;
		}
		else
		*/
		
		if(rStartTime!=0)
		{
			diff = rStopTime - rStartTime;
			rOnTime = rOnTime + diff;
		}	
  	
	}

	command uint32_t RadioAnalyzer.getOnTime() {
		
		return rOnTime;
	}

	command uint32_t RadioAnalyzer.getOffTime() {
		
		return rOffTime;
	}

	command void RadioAnalyzer.reset() {
		rOnTime = 0;
		rOffTime = 0;
	}

	command uint32_t RadioAnalyzer.getEnergyConsumed() {
		uint32_t P_on=57,P_off=8;//7.6~8;//I_on=19,I_off=2.55, V=3;
		uint64_t E;

		//P = I*V;
		E = P_on * rOnTime + P_off*rOffTime ;						
		return E;	
	}
	/************************************************************************/
	/**************************** PacketLoad *****************************/
	/************************************************************************/

	command void PacketLoad.start() {
		//call OnTimer.startPeriodic(m_rate);		
	}

	command void PacketLoad.stop() {
		call OnTimer.stop();
	}

	/*********************** Commands for PacketLoad ***************************/
		/*Monitoring Outbound traffic rate in msec*/
	command void PacketLoad.setMonitoringRate(uint16_t rate){
		m_rate = rate;
		call OnTimer.stop();		
		call OnTimer.startPeriodic(m_rate);		
	}


	/*
		Call this function every time before Interface.Send so that the sending data can be logged.		
	*/

	command void PacketLoad.logTxOut(uint8_t bytes) 
	{
		packetsSend ++;
		send_bytes = send_bytes + bytes;
		//printf("\n Txlogged with value:%d and Total:%d",bytes,send_bytes);printfflush();	
	}

	command void PacketLoad.logTxDrop(error_t err) 
	{
		if(err!=SUCCESS)
			packetsLost ++;
	}

	command void PacketLoad.logRxIn(uint8_t bytes) {
		receive_bytes = receive_bytes + bytes;
	}

	/* No. of bytes transmitted / sec*/
	command float PacketLoad.getTxDataRate()
	{
		return sentRate;
	}

	/* No. of packets dropped / sec*/
	command float PacketLoad.getTxDropRate() 
	{
		return dropRate;
	}

	/* No. of bytes received / sec*/
	command float PacketLoad.getRxDataRate() 
	{
		return receiveRate;
	}

	command float PacketLoad.getThroughputRate() 
	{
		return throughputRate;
	}

	command void PacketLoad.logPacketSend(uint8_t fType,message_t *msg) {		
		cc2420_header_t *header;
		uint8_t len;
		header = call CC2420PacketBody.getHeader(msg);
		len = call Packet.payloadLength(msg);

		PacketDelay.frameType = fType;
		PacketDelay.dsn = header->dsn;
		PacketDelay.pSendTime = call LocalTime.get();

		call PacketLoad.logTxOut(len);		
	}

	command void PacketLoad.logPacketSendDone(message_t *msg,error_t err) {
		//uint8_t len;
		//len = call Packet.getpayloadLenght(msg);

		if(err == SUCCESS)
			sent_bytes += call Packet.payloadLength(msg);

		call PacketLoad.logTxDrop(err);

		PacketDelay.error = err;
		PacketDelay.pSendDoneTime = call LocalTime.get();
	}

	command uint32_t PacketLoad.getPacketDelay() {

		uint32_t diff = 0;
		uint32_t temp = 65535L;

		/** Condition Check for Timer overflow**/
		if( PacketDelay.pSendTime > PacketDelay.pSendDoneTime){
			diff = temp - PacketDelay.pSendTime;
			diff += PacketDelay.pSendDoneTime;
		}
		else
			diff = PacketDelay.pSendDoneTime - PacketDelay.pSendTime;

		//printf("\n %u %u %u",PacketDelay.dsn,PacketDelay.error,diff);printfflush();

		return diff;
	}
	/*********************** End of PacketLoad Commands ***************************/

	/*********************** events of Timer ***************************
		Calculating Exponential weighted moving average every second.
	*/
	event void OnTimer.fired()
	{		
		refreshTxDataRate();
		refreshRxDataRate();
		refreshDropRate();
	}


	/* Calculating weighted average over outbound and inbound packets*/
	void refreshDropRate(){
		float latest = packetsLost / packetsSend *100;
		dropRate = dropRate * alpha + (1-alpha) * latest;
		//printf("\n dropRate is %d",(int)dropRate);printfflush();				
		packetsSend = 0;
		packetsLost = 0;
	}

	void refreshTxDataRate() {		
		sentRate = send_bytes * (1-alpha) + sentRate * alpha;
		//printf("\n sentRate is %d",(int)sentRate);printfflush();
		send_bytes = 0;
	}

	void refreshRxDataRate() {		
		receiveRate = receiveRate * alpha + (1-alpha) * receive_bytes;
		//printf("\n receiveRate is %d",(int)receiveRate);printfflush();
		receive_bytes = 0;		
	}

	void refreshThroughput(){
		throughputRate = throughputRate * alpha + (1- alpha)*sent_bytes;
		sent_bytes = 0;		
	}

	/************************************************************************/
	/******************************* Channel ********************************/
	/************************************************************************/




	/************************************************************************/
	/******************************** Mobility ******************************/
	/************************************************************************/


}




